Scriptname USSScript extends ReferenceAlias

WorkshopParentScript Property WorkshopParent Auto Const mandatory
GlobalVariable Property _USSEnableFood auto const
GlobalVariable Property _USSEnableWater auto const
GlobalVariable Property _USSEnableScrap auto const
GlobalVariable Property _USSEnableFertilizer auto const
GlobalVariable Property _USSEnableCaps auto const

; should be set to ChanceNone in WorkshopProduceChanceNone, so the calculations run accurately
GlobalVariable Property _USSFoodChanceNone auto const
GlobalVariable Property _USSWaterChanceNone auto const
GlobalVariable Property _USSScrapChanceNone auto const

; -1 (default) disables cap, other values
GlobalVariable Property _USSCapFood auto const
GlobalVariable Property _USSCapWater auto const
GlobalVariable Property _USSCapScrap auto const

GlobalVariable Property _USSEnableAttackReducer auto const
GlobalVariable Property _USSAttackReducerExponent auto const
GlobalVariable Property _USSMadnessMode auto const

GlobalVariable Property _USSDebugMode Auto Const

; set later to (1 - WorkshopProduceXChanceNone / 100)
float multiAddFood
float multiAddWater
float multiAddScrap

int workshopID = -1
int dailyUpdateTimerID = 1 const ; seems dumb but the vanilla script does it so whatever

ObjectReference thisWorkbench

; some constants listed verbatim from WorkshopScript
int maxStoredFoodBase = 10 const 					; stop producing when we reach this amount stored
int maxStoredFoodPerPopulation = 1 const 			; increase max for each population

int maxStoredWaterBase = 5 const 					; stop producing when we reach this amount stored
float maxStoredWaterPerPopulation = 0.25 const 		; increase max for each population

int maxStoredScavengeBase = 100 const 				; stop producing when we reach this amount stored
int maxStoredScavengePerPopulation = 5 const 		; increase max for each population

float brahminProductionBoost = 0.5 const				; what percent increase per brahmin
int maxProductionPerBrahmin = 10 const 					; each brahmin can only boost this much food (so max 10 * 0.5 = 5)
int maxBrahminFertilizerProduction = 3 const				; max fertilizer production per settlement per day
int maxStoredFertilizerBase = 10 const					; stop producing when we reach this amount stored

; vendor income
int minVendorIncomePopulation = 5 					; need at least this population to get any vendor income
float maxVendorIncome = 50.0 						; max daily vendor income from any settlement
float vendorIncomePopulationMult = 0.03 			; multiplier on population, added to vendor income
float vendorIncomeBaseMult = 2.0					; multiplier on base vendor income

float attackChanceBase = 0.02 const
float attackChanceResourceMult = 0.001 const
float attackChanceSafetyMult = 0.01 const
float attackChancePopulationMult = 0.005 const

float minProductivity = 0.25 const

bool doNextRunFood = false
bool doNextRunWater = false
bool doNextRunScavenge = false

int lastDaysSinceLastAttack = -1
int daysLeftToDelayAttacks = 0

Holotape Property _USSConfigHolotape Auto Const Mandatory
Actor Property PlayerRef Auto Const Mandatory

; Fires when the quest starts up and when it resets
Event OnAliasReset()
	thisWorkbench = getReference()

	; Only register for updates if the alias has actually been filled (I leave some aliases open for mods and DLCs)
	if (thisWorkbench != None)
		GetWorkshopID()

		; The WorkshopParent script fires this update once per day; this isn't in WorkshopScript which we're mostly emulating because WorkshopParent does it there instead
		RegisterForCustomEvent(WorkshopParent, "WorkshopDailyUpdate")

		; Debug.MessageBox("Registered workshop ID " + workshopID + " for updates")
	endif
EndEvent

Event WorkshopParentScript.WorkshopDailyUpdate(WorkshopParentScript akSender, Var[] akArgs)
	; Debug.MessageBox("caught a custom event, workshopID " + workshopID)

	float waitTime = WorkshopParent.dailyUpdateIncrement * workshopID + 0.1
	StartTimerGameTime(waitTime, dailyUpdateTimerID)
EndEvent

Event OnTimerGameTime(int aiTimerID)
	if (_USSDebugMode.getValue())
		Debug.MessageBox("USS: Caught an OnTimerGameTime event on workshop ID " + workshopID + ", running USS daily update script...")
	endif
	
	if aiTimerID == dailyUpdateTimerID
		if WorkshopParent.IsEditLocked() || WorkshopParent.DailyUpdateInProgress
			StartTimerGameTime(2.0, dailyUpdateTimerID)
		else
			DailyUpdate()
		endif
	endif
EndEvent


; So basically we have to simulate most of the calculations WorkshopScript does until DailyUpdateSurplusResources(), then work some hacky magic with the numbers that come out later
bool bDailyUpdateInProgress = false
function DailyUpdate()
	GetWorkshopID()
	
	multiAddFood = 1 - _USSFoodChanceNone.getValue() / 100
	multiAddWater = 1 - _USSWaterChanceNone.getValue() / 100
	multiAddScrap = 1 - _USSScrapChanceNone.getValue() / 100

	; wait for update lock to be released
	if bDailyUpdateInProgress || WorkshopParent.DailyUpdateInProgress
		While bDailyUpdateInProgress || WorkshopParent.DailyUpdateInProgress
			utility.wait(utility.randomFloat(0.4,0.6)) 
		EndWhile
	EndIf
	
	; tried this without the global update lock and found it would trigger the thread sync issue as with UFO4P #20295, and others
	; this probably can cause it to fire out of sync with the main update, but there's really nothing I can do about that (all it'll cause is double production on rare occasions)
	WorkshopParent.DailyUpdateInProgress = true
	bDailyUpdateInProgress = true

	; create local pointer to WorkshopRatings array to speed things up
	WorkshopDataScript:WorkshopRatingKeyword[] ratings = WorkshopParent.WorkshopRatings
	WorkshopScript:DailyUpdateData updateData = new WorkshopScript:DailyUpdateData

	; NOTE: GetBaseValue for these because we don't care if they can "produce" - actors that are wounded don't "produce" their population resource values
	updateData.totalPopulation = thisWorkbench.GetBaseValue(ratings[WorkshopParent.WorkshopRatingPopulation].resourceValue) as int
	updateData.robotPopulation = thisWorkbench.GetBaseValue(ratings[WorkshopParent.WorkshopRatingPopulationRobots].resourceValue) as int
	updateData.brahminPopulation = thisWorkbench.GetBaseValue(ratings[WorkshopParent.WorkshopRatingBrahmin].resourceValue) as int
	updateData.unassignedPopulation = thisWorkbench.GetBaseValue(ratings[WorkshopParent.WorkshopRatingPopulationUnassigned].resourceValue) as int

	updateData.vendorIncome = thisWorkbench.GetValue(ratings[WorkshopParent.WorkshopRatingVendorIncome].resourceValue) * vendorIncomeBaseMult
	updateData.currentHappiness = thisWorkbench.GetValue(ratings[WorkshopParent.WorkshopRatingHappiness].resourceValue)

	updateData.damageMult = 1 - thisWorkbench.GetValue(ratings[WorkshopParent.WorkshopRatingDamageCurrent].resourceValue)/100.0
	updateData.productivity = WorkshopParent.Workshops[workshopID].GetProductivityMultiplier(ratings)
	updateData.availableBeds = thisWorkbench.GetBaseValue(ratings[WorkshopParent.WorkshopRatingBeds].resourceValue) as int
	updateData.shelteredBeds = thisWorkbench.GetValue(ratings[WorkshopParent.WorkshopRatingBeds].resourceValue) as int
	updateData.bonusHappiness = thisWorkbench.GetValue(ratings[WorkshopParent.WorkshopRatingBonusHappiness].resourceValue) as int
	updateData.happinessModifier = thisWorkbench.GetValue(ratings[WorkshopParent.WorkshopRatingHappinessModifier].resourceValue) as int
	updateData.safety = thisWorkbench.GetValue(ratings[WorkshopParent.WorkshopRatingSafety].resourceValue) as int
	updateData.safetyDamage = thisWorkbench.GetValue(WorkshopParent.GetDamageRatingValue(ratings[WorkshopParent.WorkshopRatingSafety].resourceValue)) as int
	updateData.totalHappiness = 0.0	; sum of all happiness of each actor in town

	ObjectReference containerRef = thisWorkbench.GetContainer()
	if !containerRef
		if (_USSDebugMode.getValue())
			Debug.MessageBox("USS: Workshop ID " + workshopID + ", did not have a container (probably unowned), skipping daily update.")
		endif
	
		WorkshopParent.DailyUpdateInProgress = false
		bDailyUpdateInProgress = false
		return
	endif
	
	; Holotape loss prevention
	if (PlayerRef.GetItemCount(_USSConfigHolotape) == 0)
		if (containerRef.GetItemCount(_USSConfigHolotape) == 0)
			containerRef.AddItem(_USSConfigHolotape, 1)
		endif
	else
		if (containerRef.GetItemCount(_USSConfigHolotape) != 0)
			containerRef.RemoveItem(_USSConfigHolotape, -1)
		endif
	endif
	
	; bit of a hack but saves several hundred lines of code
	WorkshopParent.Workshops[workshopID].DailyUpdateProduceResources(ratings, updateData, containerRef, "false") 
	WorkshopParent.Workshops[workshopID].DailyUpdateConsumeResources(ratings, updateData, containerRef, "false")

	DailyUpdateSurplusResources(ratings, updateData, containerRef)
	DailyUpdateAttackLimiter(ratings, updateData)

	; clear update lock
	WorkshopParent.DailyUpdateInProgress = false
	bDailyUpdateInProgress = false
endFunction

function DailyUpdateSurplusResources(WorkshopDataScript:WorkshopRatingKeyword[] ratings, WorkshopScript:DailyUpdateData updateData, ObjectReference containerRef)
	int currentStoredFood = containerRef.GetItemCount(WorkshopParent.WorkshopConsumeFood)
	int currentStoredWater = containerRef.GetItemCount(WorkshopParent.WorkshopConsumeWater)
	int currentStoredScavenge = containerRef.GetItemCount(WorkshopParent.WorkshopConsumeScavenge)
	
	int addedFood = 0
	int addedWater = 0
	int addedScavenge = 0
	int addedFertilizer = 0
	int addedCaps = 0

	; Note: The following line comes from the base script: Bethesda is trying to run GetItemCount on a leveled list, which does not work and returns 0
	; As a result, currentStoredFertilizer is always 0, so the vanilla script does not cap it like intended and I don't have to try uncapping it
	;int currentStoredFertilizer = containerRef.GetItemCount(WorkshopParent.WorkshopProduceFertilizer)

	; In this section I do the opposite of what the vanilla script does: if I calculate that production likely ran last cycle 
	; in the vanilla script then do nothing,  and if it didn't, make up for it by producing what it would have had it ran
	
	; Approximately add the food the vanilla script didn't add, adjusting for productivity multiplier
	if _USSEnableFood.getValue() && updateData.foodProduction > 0
		bool makeFoodNow = false
		int maxFood = maxStoredFoodBase + maxStoredFoodPerPopulation * updateData.totalPopulation
		updateData.foodProduction = math.Floor(updateData.foodProduction * updateData.productivity)
		
		if updateData.foodProduction > 0
			; Is there enough food that the vanilla script has stopped producing?
			if currentStoredFood > maxFood
				; There is... but is it likely that the script already produced today? (don't want to produce twice)
				if currentStoredFood - updateData.foodProduction <= maxFood && currentStoredFood >= updateData.foodProduction * multiAddFood * 0.9
					; Yep, production most likely happened today or yesterday
					if doNextRunFood
						; The last production cycle happened yesterday - it's okay to produce today
						doNextRunFood = false
						makeFoodNow = true
					else
						; The last production cycle happened today - set a flag so it'll happen tomorrow
						doNextRunFood = true
					endif	
				else
					; We're pretty sure food production didn't happen today, so it's okay to add food
					makeFoodNow = true
					doNextRunFood = false
				endif
			else
				; the player emptied the workbench or something - clear the nextrun flag
				doNextRunFood = false
			endif
		
			if makeFoodNow
				if !(_USSCapFood.getValue() < 0) && updateData.foodProduction + currentStoredFood > _USSCapFood.getValue()
					; The user set a food cap, so modify production as needed now
					updateData.foodProduction =  Math.floor(_USSCapFood.getValue()) - currentStoredFood
				endif

				if updateData.foodProduction > 0
					WorkshopParent.ProduceFood(thisWorkbench as WorkshopScript, updateData.foodProduction)
					addedFood = updateData.foodProduction
				endif
			endif

		endif
	endif
	
	; Approximately add the water the vanilla script didn't add
	if _USSEnableWater.getValue() && updateData.waterProduction > 0
		bool makeWaterNow = false
		int maxWater =  maxStoredWaterBase + math.floor(maxStoredWaterPerPopulation * updateData.totalPopulation)
		
		; Is there enough water that the vanilla script has stopped producing?
		if currentStoredWater > maxWater
			; There is... but is it likely that the script already produced today? (don't want to produce twice)
			if currentStoredWater - updateData.waterProduction <= maxWater && currentStoredWater >= updateData.waterProduction * multiAddWater * 0.9
				; Yep, production most likely happened today or yesterday
				if doNextRunWater
					; The last production cycle happened yesterday - it's okay to produce today
					doNextRunWater = false
					makeWaterNow = true
				else
					; The last production cycle happened today - set a flag so it'll happen tomorrow
					doNextRunWater = true
				endif
			else
				; We're pretty sure water production didn't happen today, so it's okay to add water
				makeWaterNow = true
				doNextRunWater = false
			endif
		else
			; the player emptied the workbench or something - clear the nextrun flag
			doNextRunWater = false
		endif

		if makeWaterNow
			if !(_USSCapWater.getValue() < 0) && updateData.waterProduction + currentStoredWater > _USSCapWater.getValue()
				; The user set a water cap, so modify production as needed now
				updateData.waterProduction =  Math.floor(_USSCapWater.getValue()) - currentStoredWater
			endif

			if updateData.waterProduction > 0
				containerRef.AddItem(WorkshopParent.WorkshopProduceWater, updateData.waterProduction)
				addedWater = updateData.waterProduction
			endif

		endif
	endif
	
	if _USSEnableScrap.getValue()
		bool makeScavengeNow = false
		
		int scavengePopulation = (updateData.unassignedPopulation - thisWorkbench.GetValue(ratings[WorkshopParent.WorkshopRatingDamagePopulation].resourceValue)) as int
		int scavengeProductionGeneral = thisWorkbench.GetValue(ratings[WorkshopParent.WorkshopRatingScavengeGeneral].resourceValue) as int
		int scavengeAmount = math.Ceiling(scavengePopulation * updateData.productivity * updateData.damageMult + scavengeProductionGeneral*updateData.productivity)
		int maxScavenge = maxStoredScavengeBase + maxStoredScavengePerPopulation * updateData.totalPopulation
		
		if scavengeAmount > 0
			; Is there enough scavenge that the vanilla script has stopped producing?
			if currentStoredScavenge > maxScavenge
				; There is... but is it likely that the script already produced today? (don't want to produce twice)
				if (currentStoredScavenge - scavengeAmount <= maxScavenge) && (currentStoredScavenge >= scavengeAmount * multiAddScrap * 0.9)
					; Yep, production most likely happened today or yesterday
					if doNextRunScavenge
						; The last production cycle happened yesterday - it's okay to produce today
						doNextRunScavenge = false
						makeScavengeNow = true
					else
						; The last production cycle happened today - set a flag so it'll happen tomorrow
						doNextRunScavenge = true
					endif
				else
					; We're pretty sure scavenge production didn't happen today, so it's okay to add scavenge
					makeScavengeNow = true
					doNextRunScavenge = false
				endif
			else
				; the player emptied the workbench or something - clear the nextrun flag
				doNextRunScavenge = false
			endif
		
			if makeScavengeNow
				if !(_USSCapScrap.getValue() < 0) && scavengeAmount + currentStoredScavenge > _USSCapScrap.getValue()
					; The user set a scavenge cap, so modify production as needed now
					scavengeAmount =  Math.floor(_USSCapScrap.getValue()) - currentStoredScavenge
				endif

				if scavengeAmount > 0
					containerRef.AddItem(WorkshopParent.WorkshopProduceScavenge, scavengeAmount)
					addedScavenge = scavengeAmount
				endif

			endif
		endif
	endif

	; Add fertilizer past the brahmin cap (the vanilla script is uncapped anyway because of a bug)
	if _USSEnableFertilizer.getValue() && (updateData.brahminPopulation > maxBrahminFertilizerProduction)
		int fertilizerProduction = updateData.brahminPopulation - maxBrahminFertilizerProduction

		containerRef.AddItem(WorkshopParent.WorkshopProduceFertilizer, fertilizerProduction)
		addedFertilizer = fertilizerProduction
	endif

	; Add caps past cap... cap
	if updateData.vendorIncome > 0 && _USSEnableCaps.getValue()
		int vendorIncomeFinal = 0

		; get linked population with productivity excluded
		float linkedPopulation = WorkshopParent.GetLinkedPopulation(thisWorkbench as workshopScript, false)
		float vendorPopulation = linkedPopulation + updateData.totalPopulation

		; only get income if population >= minimum
		if vendorPopulation >= minVendorIncomePopulation
			linkedPopulation = WorkshopParent.GetLinkedPopulation(thisWorkbench as workshopScript, true)

			vendorPopulation = updateData.totalPopulation * updateData.productivity + linkedPopulation

			float incomeBonus = updateData.vendorIncome * vendorIncomePopulationMult * vendorPopulation

			updateData.vendorIncome = updateData.vendorIncome + incomeBonus

			vendorIncomeFinal = math.Ceiling(updateData.vendorIncome)

			; USS: add caps past the cap
			if vendorIncomeFinal > maxVendorIncome
				vendorIncomeFinal = Math.floor(vendorIncomeFinal - maxVendorIncome)
			else
				vendorIncomeFinal = 0
			endif

			if vendorIncomeFinal > 0
				containerRef.AddItem(WorkshopParent.WorkshopProduceVendorIncome, vendorIncomeFinal)
				addedCaps = vendorIncomeFinal
			endif

		endif
	EndIf

	if (_USSDebugMode.getValue())
		Debug.MessageBox("USS: Workshop ID " + workshopID +" added...\nFood: " + addedFood + "\nWater: " + addedWater + "\nScrap: " + addedScavenge + "\nFertilizer: " + addedFertilizer + "\nCaps: " + addedCaps + "\nNote that each item is a separate 75% roll, so actual created amount will be lower")
	endif
	
endFunction

Function DailyUpdateAttackLimiter(WorkshopDataScript:WorkshopRatingKeyword[] ratings, WorkshopScript:DailyUpdateData updateData)
	if !_USSEnableAttackReducer.getValue()
		; user has turned off this system, so don't even bother
		return
	endif
	
	int daysSinceLastAttack = thisWorkbench.GetValue(ratings[WorkshopParent.WorkshopRatingLastAttackDaysSince].resourceValue) as int
	int foodRating = WorkshopParent.Workshops[workshopID].GetTotalFoodRating(ratings)
	int waterRating = WorkshopParent.Workshops[workshopID].GetTotalWaterRating(ratings)

	if _USSMadnessMode.getValue()
	; Madness mode is on, so do this instead of the normal stuff
	
		if _USSMadnessMode.getValue() >= 1.9
			WorkshopParent.TriggerAttack(WorkshopParent.Workshops[workshopID], WorkshopParent.CalculateAttackStrength(foodRating, waterRating))
			
			if (_USSDebugMode.getValue())
				Debug.MessageBox("USS: Triggered an attack at workshop ID " + workshopID)
			endif
			
		elseif daysSinceLastAttack < 7
			WorkshopParent.SetResourceData(ratings[WorkshopParent.WorkshopRatingLastAttackDaysSince].resourceValue, thisWorkbench as WorkshopScript, 7)
			
			if (_USSDebugMode.getValue())
				Debug.MessageBox("USS: Reset WorkshopRatingLastAttackDaysSince at workshop ID " + workshopID)
			endif
			
		endif
		
		return
	endif
	
	if daysLeftToDelayAttacks > 0
		; basically just keep forcing LastAttackDaysSince back to 0 until we run out of "suppression days"
		daysLeftToDelayAttacks -= daysSinceLastAttack
		
		if (_USSDebugMode.getValue())
			Debug.MessageBox("USS: WorkshopID " + workshopID + " attack suppression day count now " + daysLeftToDelayAttacks + ", LADS is " +  daysSinceLastAttack + " (should be between 0-2, usually 1)")
		endif

		WorkshopParent.SetResourceData(ratings[WorkshopParent.WorkshopRatingLastAttackDaysSince].resourceValue, thisWorkbench as WorkshopScript, 0)
		
	elseif daysSinceLastAttack < lastDaysSinceLastAttack
		; only run this if we aren't already actively suppressing attacks
		; current days less than last days, which means an attack must have happened recently
		
		; calculate attack chance used in WorkshopScript
		
		float attackChance = attackChanceBase + attackChanceResourceMult * (foodRating + waterRating) - attackChanceSafetyMult*updateData.safety - attackChancePopulationMult * updateData.totalPopulation
		if attackChance < attackChanceBase
			attackChance = attackChanceBase
		endif
		
		;/ ---- 
		; Our formula here is floor(max(0,(x^0.66 + sqrt(x)) * min(1, b^0.2) * random(0.66, 1.33) - 7 - a))
		; ...where 'a' is days since last attack, x is defense and a is the daily attack chance at the moment of calculation.
		; The first part gives us a curve for defense to attack delay where doubling defense increases time between raids by about 1.5x.
		; The second part gives us a pseudo-logarithmic scale based on attack chance to mutiply the effects of the first part against, so as not to excessively reduce attacks to settlements
		; that already have a low attack chance. This kicks in like 80% when attack chance is above roughly once per week, and rapidly drops off when it's less than that. It's not perfect,
		; but it should keep attacks from being excessively reduced when the low attack chance would make them rare anyway.
		; The third part is a little random variation so that attacks won't happen on an exact schedule.
		; Next we adjust for the natural week-long cooldown on attacks, as well as days passed since the last attack happened in case this runs a day (or two+) late.
		/;
		daysLeftToDelayAttacks = Math.floor(Math.max(0, Math.pow(updateData.safety, _USSAttackReducerExponent.getValue()) + Math.sqrt(updateData.safety) * Math.min(1, Math.pow(attackChance, 0.2)) * Utility.RandomFloat(0.66, 1.33) - 7 - daysSinceLastAttack))
		
		if (_USSDebugMode.getValue())
			Debug.MessageBox("USS: Set 'suppression days' on workshopID " + workshopID + " to " + daysLeftToDelayAttacks + " because defense is " + updateData.safety + " and attackChance is " + attackChance)
		endif
	endif
	
	lastDaysSinceLastAttack = daysSinceLastAttack
EndFunction 

int function GetWorkshopID()
	if workshopID < 0
		InitWorkshopID(WorkshopParent.GetWorkshopID(thisWorkbench as WorkshopScript)) ; I can't believe this works
	endif
	return workshopID
endFunction

function InitWorkshopID(int newWorkshopID)
	if workshopID < 0
		workshopID = newWorkshopID
	endif
endFunction

