Scriptname idek:uss:USSScript extends ReferenceAlias

WorkshopParentScript Property WorkshopParent Auto Const Mandatory

; Enables the uncapping function of 
GlobalVariable Property _USSEnableFood Auto Const Mandatory
GlobalVariable Property _USSEnableWater Auto Const Mandatory
GlobalVariable Property _USSEnableScrap Auto Const Mandatory
GlobalVariable Property _USSEnableFertilizer Auto Const Mandatory
GlobalVariable Property _USSEnableCaps Auto Const Mandatory

; -1 (default) disables cap, other values set the cap on a per-settlement basis
GlobalVariable Property _USSCapFood Auto Const Mandatory
GlobalVariable Property _USSCapWater Auto Const Mandatory
GlobalVariable Property _USSCapScrap Auto Const Mandatory

GlobalVariable Property _USSEnableAttackReducer Auto Const Mandatory
GlobalVariable Property _USSAttackReducerExponent Auto Const Mandatory
GlobalVariable Property _USSMadnessMode Auto Const Mandatory

GlobalVariable Property _USSDebugMode Auto Const Mandatory

; set later to (1 - WorkshopProduceXChanceNone / 100)
float multiAddFood
float multiAddWater
float multiAddScrap

int workshopID = -1
int dailyUpdateTimerID = 1 Const
int retryRegisterTimerID = 2 Const
int dailyPreUpdateTimerID = 3 Const

ObjectReference thisWorkbench

; some constants lifted verbatim from WorkshopScript
int maxStoredFoodBase = 10 Const 					; stop producing when we reach this amount stored
int maxStoredFoodPerPopulation = 1 Const 			; increase max for each population

int maxStoredWaterBase = 5 Const 					; stop producing when we reach this amount stored
float maxStoredWaterPerPopulation = 0.25 Const 		; increase max for each population

int maxStoredScavengeBase = 100 Const 				; stop producing when we reach this amount stored
int maxStoredScavengePerPopulation = 5 Const 		; increase max for each population

float brahminProductionBoost = 0.5 Const			; what percent increase per brahmin
int maxProductionPerBrahmin = 10 Const 				; each brahmin can only boost this much food (so max 10 * 0.5 = 5)
int maxBrahminFertilizerProduction = 3 Const		; max fertilizer production per settlement per day
int maxStoredFertilizerBase = 10 Const				; stop producing when we reach this amount stored

int minVendorIncomePopulation = 5 Const				; need at least this population to get any vendor income
float maxVendorIncome = 50.0 Const					; max daily vendor income from any settlement
float vendorIncomePopulationMult = 0.03 Const		; multiplier on population, added to vendor income
float vendorIncomeBaseMult = 2.0 Const				; multiplier on base vendor income

float attackChanceBase = 0.02 Const
float attackChanceResourceMult = 0.001 Const
float attackChanceSafetyMult = 0.01 Const
float attackChancePopulationMult = 0.005 Const

float minProductivity = 0.25 Const
; end copied constants

bool firstRun = false

; used to improve the accuracy of surplus production
int lastStoredFood = 0
int lastStoredWater = 0
int lastStoredScavenge = 0
int currentDelayCount = 0
int maxDelayCount = 12 const

; used for attack reducer tracking
int lastDaysSinceLastAttack = -1 
int daysLeftToDelayAttacks = -1
int tempDaysSinceLastAttack = -1

; Make some local pointers to cut down on WorkshopParent hits and allow USS daily updates to run a few frames faster
Keyword WorkshopConsumeFood
Keyword WorkshopConsumeWater
FormList WorkshopConsumeScavenge
LeveledItem WorkshopProduceFood
LeveledItem WorkshopProduceWater
LeveledItem WorkshopProduceScavenge
LeveledItem WorkshopProduceVendorIncome
LeveledItem WorkshopProduceFertilizer

Holotape Property _USSConfigHolotape Auto Const Mandatory
Actor Property PlayerRef Auto Const Mandatory

; Fires when the quest starts up and when it resets
Event OnAliasReset()
	thisWorkbench = getReference()
	firstRun = true
		
	if (thisWorkbench != None)
		; Only register for updates if the alias has actually been filled (we leave some aliases open for mods and DLCs)
			
		if (TryRegisterForUpdates())
			; Also fill the local pointers from earlier - these are all constants, so only filling them once is fine
			WorkshopConsumeFood = WorkshopParent.WorkshopConsumeFood
			WorkshopConsumeWater = WorkshopParent.WorkshopConsumeWater
			WorkshopConsumeScavenge = WorkshopParent.WorkshopConsumeScavenge
			
			WorkshopProduceFood = WorkshopParent.WorkshopProduceFood
			WorkshopProduceWater = WorkshopParent.WorkshopProduceWater
			WorkshopProduceScavenge = WorkshopParent.WorkshopProduceScavenge
			WorkshopProduceVendorIncome = WorkshopParent.WorkshopProduceVendorIncome
			WorkshopProduceFertilizer = WorkshopParent.WorkshopProduceFertilizer
		
			if (_USSDebugMode.getValue())
				Debug.MessageBox("USS Debug: A quest alias reset and was filled.\nUSS active on workshop ID " + workshopID)
			endif
		else
			StartTimerGameTime(24.0, retryRegisterTimerID)
			
			if (_USSDebugMode.getValue())
				Debug.MessageBox("USS Debug: A quest alias reset and was filled, however a workshop ID could not be resolved. Retrying in 24 game-hours.")
			endif
		endif
	endif

EndEvent

; restore temporary changes to LADS, if necessary
Event OnAliasShutdown()
	if (tempDaysSinceLastAttack)
		LADSRestore()
	endif
EndEvent

bool Function TryRegisterForUpdates()
	GetWorkshopID()
	
	if (workshopID >= 0)
		RegisterForCustomEvent(WorkshopParent, "WorkshopDailyUpdate")
		return true
	endif
	
	return false
EndFunction

Event WorkshopParentScript.WorkshopDailyUpdate(WorkshopParentScript akSender, Var[] akArgs)
	if ((thisWorkbench as WorkshopScript).OwnedByPlayer == true)
		float waitTime = Math.Max(0, WorkshopParent.dailyUpdateIncrement * workshopID - (WorkshopParent.dailyUpdateIncrement * 0.5))
		StartTimerGameTime(waitTime, dailyPreUpdateTimerID)
	endif
EndEvent

Event OnTimerGameTime(int aiTimerID)
	if (aiTimerID == dailyUpdateTimerID)
		if ((currentDelayCount < maxDelayCount) && (lastDaysSinceLastAttack != -1) && (lastDaysSinceLastAttack == thisWorkbench.GetValue(WorkshopParent.WorkshopRatings[WorkshopParent.WorkshopRatingLastAttackDaysSince].resourceValue) as int))
			; Delay the USS update, because WorkshopScript probably hasn't run today
			
			currentDelayCount += 1
			if (_USSDebugMode.getValue())
				Debug.MessageBox("USS Debug: USS tried to run on workshop ID " + workshopID + ", but detected that WorkshopScript hasn't run yet - delaying (delay count: " + currentDelayCount + ")")
			endif
			
			StartTimerGameTime(utility.RandomFloat(0.4, 0.6), dailyUpdateTimerID)
		else			
			currentDelayCount = 0
			DailyUpdate()
		endif	
	elseif (aiTimerID == dailyPreUpdateTimerID)
		if (_USSDebugMode.getValue() > 1.9)
			Debug.MessageBox("USS Debug: Running USS pre-daily update on workshop ID " + workshopID)
		endif
		
		PreDailyUpdate()
	elseif (aiTimerID == retryRegisterTimerID)
		if (TryRegisterForUpdates())
			if (_USSDebugMode.getValue())
				Debug.MessageBox("USS Debug: A quest alias reset and was filled, and registered for updates on the second try.\nUSS active on workshop ID " + workshopID)
			endif
		else
			if (_USSDebugMode.getValue())
				Debug.MessageBox("USS Debug: A quest alias reset and was filled, however a workshop ID could not be resolved on the second try. This is normal to happen once or twice.")
			endif
		endif
	endif
EndEvent

; Collect some data before WorkshopScript runs
Function PreDailyUpdate()
	; we can accurately determine if WorkshopScript has finished running by whether LADS has changed or not
	; however, if this workshop has attacks disabled (this is normally only true for The Castle), this never changes, so special handle (read: wait longer and hope for the best)
	lastDaysSinceLastAttack = thisWorkbench.GetValue(WorkshopParent.WorkshopRatings[WorkshopParent.WorkshopRatingLastAttackDaysSince].resourceValue) as int
	
	; also, set LADS modifications here, so WorkshopScript.DailyUpdate() reads our values instead of the real LADS value,
	; and by doing so we can manipulate WorkshopScript.DailyUpdate() - these mods get restored at the end of DailyUpdateAttackLimiter()
	if (_USSEnableAttackReducer.getValue())
		if (_USSEnableAttackReducer.getValue() < 1.9)
			LADSSet(5, true)
			
			if (_USSDebugMode.getValue() > 1.9)
				Debug.MessageBox("USS Debug: WorkshopID " + workshopID + " LastAttackDaysSince rating temporarily set to 5, because attacks are disabled in config")
			endif
			
		elseif daysLeftToDelayAttacks > 0
			LADSSet(5, true)
			daysLeftToDelayAttacks -= 1
			
			if (_USSDebugMode.getValue() > 1.9)
				Debug.MessageBox("USS Debug: WorkshopID " + workshopID + " LastAttackDaysSince rating temporarily set to 5, because attack suppressor is active for this settlement - " + daysLeftToDelayAttacks + " no-attack days left")
			endif
		elseif (_USSMadnessMode.getValue())
			LADSSet(7, true)
			
			if (_USSDebugMode.getValue() > 1.9)
				Debug.MessageBox("USS Debug: WorkshopID " + workshopID + " LastAttackDaysSince rating temporarily set to 7, because madness mode enabled in config")
			endif
		endif
	endif
		
	lastStoredFood = thisWorkbench.GetItemCount(WorkshopParent.WorkshopConsumeFood)
	lastStoredWater = thisWorkbench.GetItemCount(WorkshopParent.WorkshopConsumeWater)
	lastStoredScavenge = thisWorkbench.GetItemCount(WorkshopParent.WorkshopConsumeScavenge)
	
	if ((thisWorkbench as WorkshopScript).AllowAttacks == true)	
		StartTimerGameTime(WorkshopParent.dailyUpdateIncrement*2.5, dailyUpdateTimerID) ; attacks disabled, so delay it a bit to increase reliability somewhat
	else
		StartTimerGameTime(WorkshopParent.dailyUpdateIncrement*1.25, dailyUpdateTimerID)
	endif
	
	if (_USSDebugMode.getValue())
		Debug.MessageBox("USS Debug: USS pre-daily update complete on Workshop ID " + workshopID +". Counted...\nFood: " + lastStoredFood + "\nWater: " + lastStoredWater + "\nScrap: " + lastStoredScavenge + "\nLADS: " + lastDaysSinceLastAttack)
	endif
EndFunction

; So basically we have to simulate most of the calculations WorkshopScript does until DailyUpdateSurplusResources(), then work some hacky magic with the numbers that come out later
Function DailyUpdate()
	GetWorkshopID()
	
	; wait for update lock to be released
	while (WorkshopParent.DailyUpdateInProgress)
		utility.wait(utility.randomFloat(0.4,0.6)) 
	endwhile
		
	; tried this without the global update lock and found it would trigger the thread sync issue as with UFO4P #20295, and others
	; this probably can cause it to fire out of sync with the main update, but there's really nothing I can do about that (might cause skipped or doubled production sometimes)
	WorkshopParent.DailyUpdateInProgress = true

	; create local pointer to WorkshopRatings array to speed things up
	WorkshopDataScript:WorkshopRatingKeyword[] ratings = WorkshopParent.WorkshopRatings
	WorkshopScript:DailyUpdateData updateData = new WorkshopScript:DailyUpdateData

	updateData.totalPopulation = thisWorkbench.GetBaseValue(ratings[WorkshopParent.WorkshopRatingPopulation].resourceValue) as int
	updateData.robotPopulation = thisWorkbench.GetBaseValue(ratings[WorkshopParent.WorkshopRatingPopulationRobots].resourceValue) as int
	updateData.brahminPopulation = thisWorkbench.GetBaseValue(ratings[WorkshopParent.WorkshopRatingBrahmin].resourceValue) as int
	updateData.unassignedPopulation = thisWorkbench.GetBaseValue(ratings[WorkshopParent.WorkshopRatingPopulationUnassigned].resourceValue) as int

	updateData.vendorIncome = thisWorkbench.GetValue(ratings[WorkshopParent.WorkshopRatingVendorIncome].resourceValue) * vendorIncomeBaseMult
	updateData.currentHappiness = thisWorkbench.GetValue(ratings[WorkshopParent.WorkshopRatingHappiness].resourceValue)

	updateData.damageMult = 1 - thisWorkbench.GetValue(ratings[WorkshopParent.WorkshopRatingDamageCurrent].resourceValue)/100.0
	updateData.productivity = WorkshopParent.Workshops[workshopID].GetProductivityMultiplier(ratings)
	updateData.availableBeds = thisWorkbench.GetBaseValue(ratings[WorkshopParent.WorkshopRatingBeds].resourceValue) as int
	updateData.shelteredBeds = thisWorkbench.GetValue(ratings[WorkshopParent.WorkshopRatingBeds].resourceValue) as int
	updateData.bonusHappiness = thisWorkbench.GetValue(ratings[WorkshopParent.WorkshopRatingBonusHappiness].resourceValue) as int
	updateData.happinessModifier = thisWorkbench.GetValue(ratings[WorkshopParent.WorkshopRatingHappinessModifier].resourceValue) as int
	updateData.safety = thisWorkbench.GetValue(ratings[WorkshopParent.WorkshopRatingSafety].resourceValue) as int
	updateData.safetyDamage = thisWorkbench.GetValue(WorkshopParent.GetDamageRatingValue(ratings[WorkshopParent.WorkshopRatingSafety].resourceValue)) as int
	updateData.totalHappiness = 0.0

	ObjectReference containerRef = thisWorkbench.GetContainer()
	if (!containerRef)
		if (_USSDebugMode.getValue())
			Debug.MessageBox("USS Debug: Workshop ID " + workshopID + ", did not have a container (probably unowned?), skipping daily update.")
		endif
	
		WorkshopParent.DailyUpdateInProgress = false
		return
	endif
	
	; Holotape loss prevention
	if (PlayerRef.GetItemCount(_USSConfigHolotape) == 0)
		if (containerRef.GetItemCount(_USSConfigHolotape) == 0)
			containerRef.AddItem(_USSConfigHolotape, 1)
		endif
	else
		if (containerRef.GetItemCount(_USSConfigHolotape) != 0)
			containerRef.RemoveItem(_USSConfigHolotape, -1)
		endif
	endif
	
	; bit of a hack but saves several hundred lines of code
	WorkshopParent.Workshops[workshopID].DailyUpdateProduceResources(ratings, updateData, containerRef, false) 
	WorkshopParent.Workshops[workshopID].DailyUpdateConsumeResources(ratings, updateData, containerRef, false)

	DailyUpdateSurplusResources(ratings, updateData, containerRef)
	
	if (_USSEnableAttackReducer.getValue())
		DailyUpdateAttackLimiter(ratings, updateData)
	endif

	; clear update lock
	WorkshopParent.DailyUpdateInProgress = false
	firstRun = false
EndFunction

Function DailyUpdateSurplusResources(WorkshopDataScript:WorkshopRatingKeyword[] ratings, WorkshopScript:DailyUpdateData updateData, ObjectReference containerRef)
	int currentStoredFood = containerRef.GetItemCount(WorkshopConsumeFood)
	int currentStoredWater = containerRef.GetItemCount(WorkshopConsumeWater)
	int currentStoredScavenge = containerRef.GetItemCount(WorkshopConsumeScavenge)
	
	int addedFood = 0
	int addedWater = 0
	int addedScavenge = 0
	int addedFertilizer = 0
	int addedCaps = 0

	; Note: The following line comes from the base script: Bethesda is trying to run GetItemCount on a leveled list, which does not work and returns 0
	; As a result, currentStoredFertilizer is always 0, so the vanilla script does not cap it like intended and I don't have to try uncapping it
	;int currentStoredFertilizer = containerRef.GetItemCount(WorkshopParent.WorkshopProduceFertilizer)

	; In this section I do the opposite of what the vanilla script does: if I calculate that production likely ran last cycle 
	; in the vanilla script then do nothing,  and if it didn't, make up for it by producing what it would have had it ran
	
	; Approximately add the food the vanilla script didn't add
	if ((updateData.foodProduction > 0) && (_USSEnableFood.getValue()))
		updateData.foodProduction = math.Floor(updateData.foodProduction * updateData.productivity)
		
		; Is food production still non-zero? -- is there more water than the pre-update? if so, don't produce, because WorkshopScript probably already did
		if ((updateData.foodProduction > 0) && (currentStoredFood <= lastStoredFood))
			int maxFood = maxStoredFoodBase + maxStoredFoodPerPopulation * updateData.totalPopulation
			
			; Is there enough food that the vanilla script has stopped producing?
			if (currentStoredFood > maxFood)
				; if currentStoredFood - updateData.foodProduction <= maxFood && currentStoredFood >= updateData.foodProduction * multiAddFood * 0.9

				; > _USSCapFood < 0 = unlimited
				if (!(_USSCapFood.getValue() < 0) && (updateData.foodProduction + currentStoredFood > _USSCapFood.getValue()))
					; The user set a food cap, so modify production as needed now
					updateData.foodProduction =  Math.floor(_USSCapFood.getValue()) - currentStoredFood
				endif

				if (updateData.foodProduction > 0)
					WorkshopParent.ProduceFood(thisWorkbench as WorkshopScript, updateData.foodProduction)
					addedFood = updateData.foodProduction
				endif
				
			endif
		endif
	endif	
	
	; Approximately add the water the vanilla script didn't add
	if ((updateData.waterProduction > 0) && (_USSEnableWater.getValue()))
		
		; is there more water than the pre-update? if so, don't produce, because WorkshopScript probably already did
		if (currentStoredWater <= lastStoredWater)
			int maxWater =  maxStoredWaterBase + math.floor(maxStoredWaterPerPopulation * updateData.totalPopulation)
			
			; Is there enough water that the vanilla script has stopped producing?
			if (currentStoredWater > maxWater)
				; if currentStoredWater - updateData.waterProduction <= maxWater && currentStoredWater >= updateData.waterProduction * multiAddWater * 0.9

				if (!(_USSCapWater.getValue() < 0) && (updateData.waterProduction + currentStoredWater > _USSCapWater.getValue()))
					; The user set a water cap, so modify production as needed now
					updateData.waterProduction =  Math.floor(_USSCapWater.getValue()) - currentStoredWater
				endif

				if (updateData.waterProduction > 0)
					containerRef.AddItem(WorkshopProduceWater, updateData.waterProduction)
					addedWater = updateData.waterProduction
				endif
				
			endif
		endif
	endif
	
	if (_USSEnableScrap.getValue())
	
		int scavengePopulation = (updateData.unassignedPopulation - thisWorkbench.GetValue(ratings[WorkshopParent.WorkshopRatingDamagePopulation].resourceValue)) as int
		int scavengeProductionGeneral = thisWorkbench.GetValue(ratings[WorkshopParent.WorkshopRatingScavengeGeneral].resourceValue) as int
		int scavengeAmount = math.Ceiling(scavengePopulation * updateData.productivity * updateData.damageMult + scavengeProductionGeneral*updateData.productivity)
		int maxScavenge = maxStoredScavengeBase + maxStoredScavengePerPopulation * updateData.totalPopulation
		
		if ((scavengeAmount > 0) && (currentStoredScavenge <= lastStoredScavenge) && (currentStoredScavenge > maxScavenge))
			; if (currentStoredScavenge - scavengeAmount <= maxScavenge) && (currentStoredScavenge >= scavengeAmount * multiAddScrap * 0.9)
			
			if (!(_USSCapScrap.getValue() < 0) && (scavengeAmount + currentStoredScavenge > _USSCapScrap.getValue()))
				; The user set a scavenge cap, so modify production as needed now
				scavengeAmount = Math.floor(_USSCapScrap.getValue()) - currentStoredScavenge
			endif

			if (scavengeAmount > 0)
				containerRef.AddItem(WorkshopProduceScavenge, scavengeAmount)
				addedScavenge = scavengeAmount
			endif

		endif
	endif

	; Add fertilizer past the brahmin cap (the vanilla script is uncapped anyway because of a bug)
	if (_USSEnableFertilizer.getValue() && (updateData.brahminPopulation > maxBrahminFertilizerProduction))
		int fertilizerProduction = updateData.brahminPopulation - maxBrahminFertilizerProduction

		containerRef.AddItem(WorkshopProduceFertilizer, fertilizerProduction)
		addedFertilizer = fertilizerProduction
	endif

	; Add caps past cap... cap
	if (updateData.vendorIncome > 0 && _USSEnableCaps.getValue())
		int vendorIncomeFinal = 0

		; get linked population with productivity excluded
		float linkedPopulation = WorkshopParent.GetLinkedPopulation(thisWorkbench as workshopScript, false)
		float vendorPopulation = linkedPopulation + updateData.totalPopulation

		; only get income if population >= minimum
		if (vendorPopulation >= minVendorIncomePopulation)
			linkedPopulation = WorkshopParent.GetLinkedPopulation(thisWorkbench as workshopScript, true)

			vendorPopulation = updateData.totalPopulation * updateData.productivity + linkedPopulation

			float incomeBonus = updateData.vendorIncome * vendorIncomePopulationMult * vendorPopulation

			updateData.vendorIncome = updateData.vendorIncome + incomeBonus

			vendorIncomeFinal = math.Ceiling(updateData.vendorIncome)

			; USS: add caps past the cap
			if (vendorIncomeFinal > maxVendorIncome)
				vendorIncomeFinal = Math.floor(vendorIncomeFinal - maxVendorIncome)
			else
				vendorIncomeFinal = 0
			endif

			if (vendorIncomeFinal > 0)
				containerRef.AddItem(WorkshopProduceVendorIncome, vendorIncomeFinal)
				addedCaps = vendorIncomeFinal
			endif

		endif
	endif

	if (_USSDebugMode.getValue())
		Debug.MessageBox("USS Debug: USS daily update complete on Workshop ID " + workshopID +". Added...\nFood: " + addedFood + "\nWater: " + addedWater + "\nScrap: " + addedScavenge + "\nFertilizer: " + addedFertilizer + "\nCaps: " + addedCaps)
	endif
	
EndFunction

Function DailyUpdateAttackLimiter(WorkshopDataScript:WorkshopRatingKeyword[] ratings, WorkshopScript:DailyUpdateData updateData)
	int daysSinceLastAttack = thisWorkbench.GetValue(ratings[WorkshopParent.WorkshopRatingLastAttackDaysSince].resourceValue) as int
	
	if (_USSMadnessMode.getValue() >= 1.9)
		int foodRating = WorkshopParent.Workshops[workshopID].GetTotalFoodRating(ratings)
		int waterRating = WorkshopParent.Workshops[workshopID].GetTotalWaterRating(ratings)
	
		WorkshopParent.TriggerAttack(WorkshopParent.Workshops[workshopID], WorkshopParent.CalculateAttackStrength(foodRating, waterRating))
		
		if (_USSDebugMode.getValue())
			Debug.MessageBox("USS Debug: Triggered an attack at workshop ID " + workshopID + " because extra madness mode enabled")
		endif
	endif
	
	if ((daysSinceLastAttack < lastDaysSinceLastAttack && tempDaysSinceLastAttack == -1) || firstRun)
		; current LADS less than last LADS, and there's no temporary change to LADS at this settlement, which means an attack must have happened today
		; alternatively this is the first run, so run this block even though there's no attack
	
		int foodRating = WorkshopParent.Workshops[workshopID].GetTotalFoodRating(ratings)
		int waterRating = WorkshopParent.Workshops[workshopID].GetTotalWaterRating(ratings)
		
		; calculate attack chance used in WorkshopScript
		float attackChance = attackChanceBase + attackChanceResourceMult * (foodRating + waterRating) - attackChanceSafetyMult * updateData.safety - attackChancePopulationMult * updateData.totalPopulation
		
		if (attackChance < attackChanceBase)
			attackChance = attackChanceBase
		endif
		
		;/ ---- 
		; Our formula here is floor(max(0,(x^0.66 + sqrt(x)) * min(1, b^0.2) * random(0.66, 1.33) - a)) ---- ( <= 1.3.1 formula also subtracted 7 from a)
		; ...where 'a' is days since last attack, x is defense and b is the daily attack chance at the moment of calculation.
		; The first part gives us a curve for defense to attack delay where doubling defense increases time between raids by about 1.5x.
		; The second part gives us a pseudo-logarithmic scale based on attack chance to mutiply the effects of the first part against, so as not to excessively reduce attacks to settlements
		; that already have a low attack chance. This kicks in like 80% when attack chance is above roughly once per week, and rapidly drops off when it's less than that. It's not perfect,
		; but it should keep attacks from being excessively reduced when the low attack chance would make them rare anyway.
		; The third part is a little random variation so that attacks won't happen on an exact schedule.
		; Last, we adjust for days passed since the last attack happened in case this runs a day (or two+) late, make sure our value isn't negative, and round it down into an int
		/;
		daysLeftToDelayAttacks = Math.floor(Math.max(0, Math.pow(updateData.safety, _USSAttackReducerExponent.getValue()) + Math.sqrt(updateData.safety) * Math.min(1, Math.pow(attackChance, 0.2)) * Utility.RandomFloat(0.66, 1.33) - daysSinceLastAttack))
		
		if (firstRun)
			; this is the first run, so multiply daysLeftToDelayAttacks by a random amount
			; this won't really do anything on new games, but will affect existing games, and in particular those where USS is just being reset
			daysLeftToDelayAttacks = Math.floor(daysLeftToDelayAttacks * Utility.RandomFloat(0, 1))
			
			if (_USSDebugMode.getValue())
				Debug.MessageBox("USS Debug: First Attack Reducer run at Workshop ID " + workshopID + "\n\nSet 'suppression days' to " + daysLeftToDelayAttacks + " because defense is " + updateData.safety + " and attackChance is " + attackChance + ", and this is the first run")
			endif
		else
			if (_USSDebugMode.getValue())
				Debug.MessageBox("USS Debug: Attack detected at Workshop ID " + workshopID + "\n\nSet 'suppression days' to " + daysLeftToDelayAttacks + " because defense is " + updateData.safety + " and attackChance is " + attackChance)
			endif
		endif
	else
		; reset LADS on workbench to value from before PreDailyUpdate(), and add a day
		; only do this if there was no attack today
		LADSRestore(daysSinceLastAttack - tempDaysSinceLastAttack)
	endif
EndFunction 

;/ Why do I do it this way, by hacking DaysSinceLastAttack instead of just disabling attacks using the AllowAttacks script property?
; This way wears off over time. If USS gets uninstalled, or reset, or whatever, I -really- do not want to permanently break attacks on
; all affected settlements, and it's likely that vanilla scripts fiddle with the AllowAttacks property sometimes, which could also break things.
; Doing it this way, if USS gets uninstalled, or reset, or stops working (etc), worst-case-scenario is that the save goes back to normal
; after one game-week.
; 
; I know it's hacky, and might mildly conflict with e.g. Settlement Management Software, but it really is preferable to alternatives
/;

; set LastAttackDaysSince, with or without provisions for setting it back in DailyUpdate()
Function LADSSet(int i, bool restoreLater = false)
	WorkshopParent.SetResourceData(WorkshopParent.WorkshopRatings[WorkshopParent.WorkshopRatingLastAttackDaysSince].resourceValue, thisWorkbench as WorkshopScript, i as float)
	
	if (restoreLater)
		tempDaysSinceLastAttack = i
	else
		tempDaysSinceLastAttack = -1
	endif
	
	;if (_USSDebugMode.getValue() > 1.9)
	;	Debug.MessageBox("USS Debug: LADS set to a value of " + thisWorkbench.GetValue(WorkshopParent.WorkshopRatings[WorkshopParent.WorkshopRatingLastAttackDaysSince].resourceValue) as int + ", restoreLater = " + restoreLater)
	;endif
EndFunction

; reset LastAttackDaysSince to the last value set by LADSSet(), plus whatever value it gets passed
Function LADSRestore(int i = 0)
	if (tempDaysSinceLastAttack != -1)
		LADSSet(lastDaysSinceLastAttack + i, false)
	endif
	
	;if (_USSDebugMode.getValue() > 1.9)
	;	Debug.MessageBox("USS Debug: LADS restored to a value of " + thisWorkbench.GetValue(WorkshopParent.WorkshopRatings[WorkshopParent.WorkshopRatingLastAttackDaysSince].resourceValue) as int)
	;endif
EndFunction

int Function GetWorkshopID()
	if (workshopID < 0)
		workshopID = WorkshopParent.GetWorkshopID(thisWorkbench as WorkshopScript)
		
		;if (workshopID < 0 && _USSDebugMode.getValue()); couldn't get workshop ID
		;	Debug.MessageBox("USS Debug: A workshop was unable to get an ID from WorkshopParent - object is " + thisWorkbench)
		;endif
	endif
	
	return workshopID
EndFunction
